/** @file   lightfence.h
 * @brief   Declaration of LightFence - class
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:38 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_ENG2D_LIGHTFENCE_H
#define H_ENG2D_LIGHTFENCE_H

#include <vector>
#include "eng2d_dll.h"
#include "interactive.h"
#include "wire.h"
#include "sound.h"

namespace eng2d {

/** @class  LightFence
 * @brief   Represents a lightfence or electric fence.
 * @author  Tomi Lamminsaari
 *
 */
class DLLIMPORT LightFence : public Interactive
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  typedef int State;
  static const State STATE_ON = 1;
  static const State STATE_OFF = 0;
  
  static const int FENCETYPE_LIGHTFENCE = 0;
  

  /** Defines the state and duration of it. With several of these structures
   * applied one after another we can create state sequences where the gate
   * is on and off with different times. Use @c setStateSequence() - method
   * to set the sequences this fence operates.
   */
  struct StateProp {
    /** The state */
    State state;
    /** How long this state should last. */
    int duration;
  };
  
  
  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   * @param     rP1               The beginning point coordinate
   * @param     rP2               The end point coordinate
   */
	LightFence( const Vec2D& rP1, const Vec2D& rP2 );


	/** Destructor
   */
	virtual ~LightFence();

private:

	/** Copy constructor.
   * @param     rO                Reference to another LightFence
   */
  LightFence( const LightFence& rO );

	/** Assignment operator
   * @param     rO                Reference to another LightFence
   * @return    Reference to us.
   */
  LightFence& operator = ( const LightFence& rO );

public:


  ///
  /// Methods
  /// =======

  /** Updates this fence
   */
  virtual void update();
  
  /** Draws this fence
   * @param     pB                Pointer to target bitmap.
   * @param     offset            An offset needed when translating from
   *                              world-coordinates to screen coordinates.
   */
  virtual void redraw( BITMAP* pB, const Vec2D& offset ) const;


  /** Sets the operation sequences. Once the state sequence has been played
   * through we loop it again.
   * @param     stateSequence     A vector that holds the StateProp - structures
   *                              that form the sequence how this fence
   *                              operates. If this is an empty vector,
   *                              the fence gets to off-mode.
   */
  virtual void setStateSequence( const std::vector<StateProp>& stateSequence );
  
  /** Attaches the given SoundSource to this fence. This method adds the
   * given SoundSource to Sound - class so it comes hearable right away.
   * The ownership of the given SoundSource is handed to Sound - class so
   * this class as well as the caller must not delete it.
   *
   * If you reattach a new SoundSource, the previously given SoundSource gets
   * destroyed.
   *
   * To destroy the SoundSource previously attached to us, call this method
   * by passing a nullpointer.
   * @param     pSS               Pointer to SoundSource.
   */
  virtual void attachSoundSource( SoundSource* pSS );
  
  
  
  ///
  /// Getter methods
  /// ==============

  /** Tells if the given point is between the beginning node and ending
   * node of this fence. If the fence is in STATE_OFF - mode, this
   * method returns false.
   * @param     rP                The point
   * @param     maxDistance       The maximum distance from the fence-line
   *                              that still touches the fence.
   * @return    <code>true</code> if given point is touching this fence.
   *            If state is STATE_OFF this returns @c false.
   */
  virtual bool touches( const Vec2D& rP, float maxDistance ) const;
  
  /** Returns the state of this fence
   * @return    State of this fence.
   */
  virtual State state() const;
  
  /** Returns the pointer to SoundSource that is attached to this fence.
   * @return    Pointer to SoundSource or nullpointer if this fence does
   *            not have SoundSource.
   */
  virtual SoundSource* getSoundSource() const;
  
  /** Returns the position of the first end of this fence.
   * @return    Coordinate of the first end of this fence.
   */
  virtual Vec2D nodePos1() const;
  
  /** Returns the position of the second end of this fence.
   * @return    Position of the second end.
   */
  virtual Vec2D nodePos2() const;
  
  /** Returns the coordinate of the middle point
   * @return    Returns the middlepoint coordinate.
   */
  virtual Vec2D middlePos() const;
  
  /** Returns the type id of this fence. By default we return 0.
   * @return    Type id 0
   */
  virtual int fenceType() const;

protected:

  /** Adds new Wire to this fence. The ownership of the given Wire is
   * given to us so this LightFence will take care of destroying it.
   * @param     pW                Pointer to Wire being added here.
   */
  void addWire( Wire* pW );
  

  ///
  /// Members
  /// =======

  /** The beginning coordinate of the fence */
  Vec2D m_begPos;
  /** The ending coordinate of the fence */
  Vec2D m_endPos;
  
  /** The reference points we use when finding if player touches this fence. */
  std::vector< Vec2D > m_refpoints;
  
  /** The state of this fence. */
  State m_state;
  
  /** An ID-code of this fence. */
  int m_idcode;
  
  /** The duration of on-state. */
  int m_onStateLength;
  /** The duration of off-state */
  int m_offStateLength;
  
  /** A counter that calculates the duration of the states. */
  int m_stateCounter;

  /** This vector holds the wires we use for the fence. */
  std::vector< Wire* >  m_wires;

  /** The state sequence we're playing. */
  std::vector< StateProp > m_stateSequence;
  
  /** The position in m_stateSequence - vector we're current playing. */
  std::vector< StateProp >::iterator m_stateIter;
  
  /** Pointer to possible SoundSource - object. */
  SoundSource* m_pSound;

private:

  ///
  /// Private members
  /// ===============


};

};  // end of namespace
#endif
